<?php

namespace App\Imports;

use App\Models\Product;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;

class ProductsImport implements ToModel, WithBatchInserts, WithChunkReading, WithHeadingRow, WithValidation
{
    private $categoryId;

    private $source;

    public function __construct($categoryId = null, $source = 'aliexpress')
    {
        $this->categoryId = $categoryId;
        $this->source = $source;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        return new Product([
            'category_id' => $this->categoryId,
            'source' => $this->source,
            'name_ar' => $row['product_desc'] ?? 'No description',
            'name_en' => $row['product_desc'] ?? 'No description',
            'description_ar' => $row['product_desc'] ?? null,
            'description_en' => $row['product_desc'] ?? null,
            'price' => $row['origin_price'] ?? 0,
            'discount_price' => $row['discount_price'] ?? null,
            'currency' => $row['currency'] ?? 'EGP',
            'clicks' => 0,
            'direct_commission_rate' => $row['direct_linking_commission_rate_'] ?? 0,
            'estimated_direct_commission' => $row['estimated_direct_linking_commission'] ?? 0,
            'indirect_commission_rate' => $row['indirect_linking_commission_rate_'] ?? 0,
            'estimated_indirect_commission' => $row['estimated_indirect_linking_commission'] ?? 0,
            'sales_180day' => $row['sales180day'] ?? 0,
            'positive_feedback' => $row['positive_feedback'] ?? 0,
            'promotion_url' => $row['promotion_url'] ?? null,
            'code_name' => $row['code_name'] ?? null,
            'code_start_time' => $this->parseDate($row['code_start_time'] ?? null),
            'code_end_time' => $this->parseDate($row['code_end_time'] ?? null),
            'code_value' => $row['code_value'] ?? null,
            'code_quantity' => $row['code_quantity'] ?? null,
            'code_minimum_spend' => $row['code_minimum_spend'] ?? null,
        ]);
    }

    /**
     * Parse date from Excel
     */
    private function parseDate($date)
    {
        if (empty($date)) {
            return null;
        }

        // If it's already a DateTime object
        if ($date instanceof \DateTime) {
            return $date->format('Y-m-d H:i:s');
        }

        // Try to parse the date
        try {
            return \Carbon\Carbon::parse($date)->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Validation rules
     */
    public function rules(): array
    {
        return [
            'origin_price' => 'required|numeric',
            'currency' => 'in:EGP,USD,SAR,AED',
        ];
    }

    /**
     * Batch size
     */
    public function batchSize(): int
    {
        return 100;
    }

    /**
     * Chunk size
     */
    public function chunkSize(): int
    {
        return 100;
    }
}
