<?php

namespace Tests\Feature\Admin;

use App\Models\Category;
use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AdminProductControllerTest extends TestCase
{
    use RefreshDatabase;

    private User $admin;

    protected function setUp(): void
    {
        parent::setUp();

        $this->admin = User::factory()->create([
            'is_admin' => true,
            'admin_approved_at' => now(),
        ]);

        $this->actingAs($this->admin);
    }

    public function test_admin_can_view_products_list(): void
    {
        Product::factory()->count(5)->create();

        $response = $this->get(route('admin.products.index'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.products.index');
        $response->assertViewHas('products');
    }

    public function test_admin_can_view_product_creation_form(): void
    {
        $response = $this->get(route('admin.products.create'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.products.create');
        $response->assertViewHas('categories');
    }

    public function test_admin_can_create_product(): void
    {
        $category = Category::factory()->create();

        $productData = [
            'category_id' => $category->id,
            'source' => 'aliexpress',
            'name_ar' => 'منتج تجريبي',
            'name_en' => 'Test Product',
            'description_ar' => 'وصف المنتج',
            'description_en' => 'Product description',
            'price' => 100.00,
            'currency' => 'EGP',
        ];

        $response = $this->post(route('admin.products.store'), $productData);

        $response->assertRedirect(route('admin.products.index'));
        $response->assertSessionHas('success');
        $this->assertDatabaseHas('products', [
            'name_en' => 'Test Product',
            'price' => 100.00,
        ]);
    }

    public function test_admin_cannot_create_product_without_required_fields(): void
    {
        $response = $this->post(route('admin.products.store'), []);

        $response->assertSessionHasErrors(['category_id', 'source', 'name_ar', 'name_en', 'price', 'currency']);
    }

    public function test_admin_can_view_product_details(): void
    {
        $product = Product::factory()->create();

        $response = $this->get(route('admin.products.show', $product));

        $response->assertStatus(200);
        $response->assertViewIs('admin.products.show');
        $response->assertViewHas('product');
    }

    public function test_admin_can_edit_product(): void
    {
        $product = Product::factory()->create();
        $category = Category::factory()->create();

        $response = $this->get(route('admin.products.edit', $product));

        $response->assertStatus(200);
        $response->assertViewIs('admin.products.edit');
        $response->assertViewHas('categories');
    }

    public function test_admin_can_update_product(): void
    {
        $product = Product::factory()->create();
        $category = Category::factory()->create();

        $updatedData = [
            'category_id' => $category->id,
            'source' => 'amazon',
            'name_ar' => 'منتج محدث',
            'name_en' => 'Updated Product',
            'description_ar' => 'وصف محدث',
            'description_en' => 'Updated description',
            'price' => 200.00,
            'currency' => 'USD',
        ];

        $response = $this->put(route('admin.products.update', $product), $updatedData);

        $response->assertRedirect(route('admin.products.index'));
        $response->assertSessionHas('success');
        $this->assertDatabaseHas('products', [
            'id' => $product->id,
            'name_en' => 'Updated Product',
            'price' => 200.00,
        ]);
    }

    public function test_admin_can_delete_product(): void
    {
        $product = Product::factory()->create();

        $response = $this->delete(route('admin.products.destroy', $product));

        $response->assertRedirect(route('admin.products.index'));
        $response->assertSessionHas('success');
        $this->assertDatabaseMissing('products', [
            'id' => $product->id,
        ]);
    }

    public function test_admin_can_view_import_page(): void
    {
        $response = $this->get(route('admin.products.import'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.products.import');
    }

    public function test_admin_can_filter_products_by_category(): void
    {
        $category = Category::factory()->create();
        Product::factory()->create(['category_id' => $category->id]);
        Product::factory()->create();

        $response = $this->get(route('admin.products.index', ['category' => $category->id]));

        $response->assertStatus(200);
        $response->assertViewHas('products');
        $this->assertEquals(1, $response->viewData('products')->count());
    }

    public function test_admin_can_search_products(): void
    {
        Product::factory()->create(['name_en' => 'iPhone 15']);
        Product::factory()->create(['name_en' => 'Samsung Galaxy']);

        $response = $this->get(route('admin.products.index', ['search' => 'iPhone']));

        $response->assertStatus(200);
        $products = $response->viewData('products');
        $this->assertEquals(1, $products->count());
        $this->assertEquals('iPhone 15', $products->first()->name_en);
    }
}
